from fastapi import FastAPI, Request, Form, Depends, HTTPException
from fastapi.responses import HTMLResponse, RedirectResponse
from fastapi.templating import Jinja2Templates
from sqlalchemy.orm import Session
from database import SessionLocal, engine, Base
from models import User, ValidationLog
from auth import hash_password, verify_password, generate_api_key
from validators import validate_email_address, validate_phone_number
from config import COST_PER_VALIDATION

app = FastAPI()
templates = Jinja2Templates(directory="templates")

Base.metadata.create_all(bind=engine)

def get_db():
    db = SessionLocal()
    try:
        yield db
    finally:
        db.close()

# Home
@app.get("/", response_class=HTMLResponse)
def home(request: Request):
    return templates.TemplateResponse("index.html", {"request": request})

# Register
@app.post("/register")
def register(email: str = Form(...), password: str = Form(...), db: Session = Depends(get_db)):
    user = User(email=email,
                password=hash_password(password),
                api_key=generate_api_key())
    db.add(user)
    db.commit()
    return RedirectResponse("/login", status_code=302)

# Login
@app.post("/login")
def login(email: str = Form(...), password: str = Form(...), db: Session = Depends(get_db)):
    user = db.query(User).filter(User.email == email).first()
    if not user or not verify_password(password, user.password):
        return RedirectResponse("/login", status_code=302)
    response = RedirectResponse("/dashboard", status_code=302)
    response.set_cookie("user_id", str(user.id))
    return response

# Dashboard
@app.get("/dashboard", response_class=HTMLResponse)
def dashboard(request: Request, db: Session = Depends(get_db)):
    user_id = request.cookies.get("user_id")
    if not user_id:
        return RedirectResponse("/login", status_code=302)

    user = db.query(User).get(int(user_id))
    return templates.TemplateResponse("dashboard.html", {
        "request": request,
        "user": user
    })

# API VALIDATION
@app.post("/api/validate")
def api_validate(request: Request, input_data: str, type: str, db: Session = Depends(get_db)):

    api_key = request.headers.get("X-API-KEY")
    user = db.query(User).filter(User.api_key == api_key).first()

    if not user:
        raise HTTPException(status_code=403, detail="Invalid API Key")

    if user.credits < COST_PER_VALIDATION:
        raise HTTPException(status_code=402, detail="Insufficient Credits")

    if type == "email":
        result = validate_email_address(input_data)
    else:
        result = validate_phone_number(input_data)

    user.credits -= COST_PER_VALIDATION
    db.commit()

    return result
